<?php
session_start();

function validate_user_id($userId)
{
    return preg_match('/^[a-f0-9]{32}$/', $userId);
}

$userId = $_SESSION['user_id'] ?? '';
if (!validate_user_id($userId)) {
    header("Location: login.php");
    exit;
}

$userDir = __DIR__ . "/users/" . basename($userId);

if (!is_dir($userDir)) {
    header("Location: login.php");
    exit;
}

function safe_json_read($file)
{
    if (!file_exists($file) || !is_readable($file)) {
        return null;
    }

    $content = file_get_contents($file);
    if ($content === false) {
        return null;
    }

    $data = json_decode($content, true);
    return json_last_error() === JSON_ERROR_NONE ? $data : null;
}

$dataFile = "$userDir/data.json";
$data = safe_json_read($dataFile);

if (!$data) {
    header("Location: login.php");
    exit;
}

$username = htmlspecialchars($data['username'] ?? '', ENT_QUOTES, 'UTF-8');
$email = htmlspecialchars($data['email'] ?? '', ENT_QUOTES, 'UTF-8');
$language = htmlspecialchars($data['language'] ?? 'English', ENT_QUOTES, 'UTF-8');
$lastLogin = htmlspecialchars($data['last_login'] ?? 'Unknown', ENT_QUOTES, 'UTF-8');
$numericId = htmlspecialchars($data['user_id'] ?? $userId, ENT_QUOTES, 'UTF-8');
$displayName = htmlspecialchars($data['displayname'] ?? $username, ENT_QUOTES, 'UTF-8');

$ageCategories = [
    ['label' => 'Young', 'range' => '13-24 years'],
    ['label' => 'Adult', 'range' => '25-40 years'],
    ['label' => 'Experienced', 'range' => '41-60 years'],
    ['label' => 'Elder', 'range' => '60+ years']
];

$currentAgeCategory = htmlspecialchars($data['age'] ?? 'Young', ENT_QUOTES, 'UTF-8');
$currentIndex = array_search($currentAgeCategory, array_column($ageCategories, 'label'));
if ($currentIndex === false) $currentIndex = 0;

$lockFile = "$userDir/age_lock";
$canChangeAge = true;

if (file_exists($lockFile)) {
    $lockTime = filemtime($lockFile);
    if ((time() - $lockTime) < 86400) {
        $canChangeAge = false;
    }
}

function get_secret_key()
{
    $path = __DIR__ . "/users/secret_key.php";
    if (!file_exists($path)) {
        die("Secret key file missing!");
    }
    $data = include $path;
    return $data['key'] ?? null;
}

function encrypt_data($plaintext)
{
    $key = get_secret_key();
    $iv = random_bytes(16);
    $cipher = openssl_encrypt($plaintext, "AES-256-CBC", $key, 0, $iv);
    return base64_encode($iv . $cipher);
}

function decrypt_data($encoded)
{
    $key = get_secret_key();
    $data = base64_decode($encoded);
    $iv = substr($data, 0, 16);
    $cipher = substr($data, 16);
    return openssl_decrypt($cipher, "AES-256-CBC", $key, 0, $iv);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $requestTime = $_SERVER['REQUEST_TIME'] ?? time();
    if ($requestTime - time() > 30) {
        header("Location: settings.php");
        exit;
    }

    if (isset($_POST['displayname'])) {
        $newDisplayName = trim($_POST['displayname']);
        if ($newDisplayName && preg_match('/^[a-zA-Z0-9_\s]{1,24}$/', $newDisplayName) && $newDisplayName !== ($data['displayname'] ?? '')) {
            $data['displayname'] = $newDisplayName;
            file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
        }
    }

    if (isset($_POST['age']) && $canChangeAge) {
        $ageCategory = $_POST['age'];
        $validCategories = array_column($ageCategories, 'label');
        if (in_array($ageCategory, $validCategories)) {
            $data['age'] = $ageCategory;
            file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
            touch($lockFile);
            $canChangeAge = false;
            $currentIndex = array_search($ageCategory, $validCategories);
        }
    }

    if (isset($_POST['change_password'])) {
        $currentPass = $_POST['current_password'] ?? '';
        $newPass = $_POST['new_password'] ?? '';
        $repeatPass = $_POST['repeat_password'] ?? '';

        $authFile = $userDir . "/auth.enc";

        if (!file_exists($authFile)) {
            return;
        } else {
            $encryptedData = file_get_contents($authFile);
            $decryptedData = decrypt_data($encryptedData);

            if ($decryptedData === false) {
                return;
            } else {
                $parts = explode("|", $decryptedData);
                if (count($parts) === 2) {
                    list($storedUsername, $storedHash) = $parts;

                    if (!password_verify($currentPass, $storedHash)) {
                        $passwordError = "Current password is incorrect";
                    } elseif ($newPass !== $repeatPass) {
                        $passwordError = "New passwords do not match";
                    } elseif (strlen($newPass) < 8) {
                        $passwordError = "Password must be at least 8 characters long";
                    } elseif (preg_match('/\s/', $newPass) || !preg_match('/^[a-zA-Z0-9_]+$/', $newPass)) {
                        $passwordError = "Password can only contain English letters, numbers, and underscore, no spaces allowed";
                    } else {
                        $newHash = password_hash($newPass, PASSWORD_DEFAULT);
                        $newAuthData = $username . "|" . $newHash;
                        $newEncrypted = encrypt_data($newAuthData);

                        if ($newEncrypted === false) {
                            return;
                        } else {
                            if (file_put_contents($authFile, $newEncrypted, LOCK_EX) !== false) {
                                $passwordSuccess = "Password changed successfully!";
                            } else {
                                return;
                            }
                        }
                    }
                } else {
                    return;
                }
            }
        }
    }
}
?>

<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="/images/favicon.png">
    <title>Rankly / Settings</title>
</head>

<body>
    <?php include "header.php"; ?>

    <div class="mt-4">
        <div class="flex text-white items-center justify-between mb-4 ml-4">
            <div class="flex items-center gap-3">
                <a href="profile.php" class="text-white transition">
                    <i data-lucide="arrow-left" class="w-5 h-5"></i>
                </a>
                <h1 class="text-xl font-semibold">Settings</h1>
            </div>
        </div>
        <div class="border-b border-white/20"></div>

        <div class="flex bg-[#ffffff05] p-4 items-left justify-start flex-col">
            <h2 class="text-lg text-white font-semibold mb-3">Account Information</h2>
            <div class="space-y-2 text-[#ffffff95] text-sm">
                <div class="flex justify-between">
                    <span class="text-[#ffffff95]">Username</span>
                    <span class="font-medium">@<?= $username ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-[#ffffff95]">Email</span>
                    <span class="font-medium"><?= $email ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-[#ffffff95]">Language</span>
                    <span class="font-medium"><?= $language ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-[#ffffff95]">User ID</span>
                    <span class="font-medium"><?= $numericId ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-[#ffffff95]">Last Login</span>
                    <span class="font-medium"><?= $lastLogin ?></span>
                </div>
            </div>
        </div>

        <div class="mt-5 mx-4">
            <h2 class="text-lg font-semibold text-white mb-3">Display Name</h2>
            <div class="flex items-center gap-2">
                <input id="displayNameInput" type="text"
                    maxlength="24"
                    value="<?= $displayName ?>"
                    placeholder="Enter display name"
                    class="flex-1 bg-[#2a2a2a] text-white rounded-xl px-4 py-2 focus:outline-none focus:ring-2 focus:ring-[#3b82f6] placeholder-[#ffffff50] transition" />

                <button id="saveDisplayNameBtn"
                    class="bg-white w-10 h-10 flex items-center justify-center rounded-xl transition">
                    <i data-lucide="check" class="w-5 h-5 text-black"></i>
                </button>
            </div>
        </div>

        <div class="mt-5 mx-4">
            <h2 class="text-lg font-semibold text-white mb-3">Select Your Age Category</h2>
            <div class="flex items-center justify-center">
                <button id="prevAgeBtn" class="bg-[#2a2a2a] text-white text-black w-10 h-10 flex items-center justify-center rounded-l-xl">
                    <i data-lucide="chevron-left" class="w-6 h-6"></i>
                </button>

                <div id="ageDisplayBox" class="bg-[#2a2a2a] text-white flex flex-col items-center justify-center w-full h-10 transition-all duration-300">
                    <span id="ageLabel" class="text-[12px] font-semibold"><?= $ageCategories[$currentIndex]['label'] ?></span>
                    <span id="ageRange" class="text-[10px]"><?= $ageCategories[$currentIndex]['range'] ?></span>
                </div>

                <button id="nextAgeBtn" class="bg-[#2a2a2a] text-white text-black w-10 h-10 flex items-center justify-center rounded-r-xl">
                    <i data-lucide="chevron-right" class="w-6 h-6"></i>
                </button>
            </div>

            <form id="ageForm" method="POST" class="mt-4">
                <input type="hidden" name="age" id="ageCategoryInput" value="<?= $ageCategories[$currentIndex]['label'] ?>">
                <button type="submit" <?= $canChangeAge ? '' : 'disabled' ?>
                    class="bg-white text-black w-full py-2 rounded-xl transition <?= $canChangeAge ? '' : 'opacity-50 cursor-not-allowed' ?>">
                    <?= $canChangeAge ? 'Save Age Category' : 'Age Locked for 24h' ?>
                </button>
            </form>
        </div>

        <div class="mt-5 mx-4">
            <h2 class="text-lg font-semibold text-white mb-3">Change Password</h2>
            <form id="passwordForm" method="POST" class="flex flex-col gap-3 relative">
                <input type="hidden" name="change_password" value="1">

                <div class="relative">
                    <input type="password" name="current_password" placeholder="Current Password" required
                        class="w-full bg-[#2a2a2a] text-white rounded-xl px-4 py-2 focus:outline-none transition">
                </div>

                <div class="relative">
                    <input type="password" id="new_password" name="new_password" placeholder="New Password" required minlength="8"
                        class="w-full bg-[#2a2a2a] text-white rounded-xl px-4 py-2 focus:outline-none transition pr-10">
                    <button type="button" id="showNewBtn"
                        class="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 flex items-center justify-center">
                        <i data-lucide="eye" class="w-4 h-4 text-[#ffffff85]"></i>
                    </button>
                </div>

                <div class="relative">
                    <input type="password" id="repeat_password" name="repeat_password" placeholder="Repeat New Password" required minlength="8"
                        class="w-full bg-[#2a2a2a] text-white rounded-xl px-4 py-2 focus:outline-none transition pr-10">
                    <button type="button" id="showRepeatBtn"
                        class="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 flex items-center justify-center">
                        <i data-lucide="eye" class="w-4 h-4 text-[#ffffff85]"></i>
                    </button>
                </div>

                <?php if ($passwordError): ?>
                    <div class="text-red-400 text-sm text-center py-2">
                        <?= htmlspecialchars($passwordError, ENT_QUOTES, 'UTF-8') ?>
                    </div>
                <?php endif; ?>

                <?php if ($passwordSuccess): ?>
                    <div class="text-green-400 text-sm text-center py-2">
                        <?= htmlspecialchars($passwordSuccess, ENT_QUOTES, 'UTF-8') ?>
                    </div>
                <?php endif; ?>

                <button type="submit" class="bg-white text-black w-full py-2 rounded-xl transition">Confirm</button>
            </form>
        </div>
    </div>

    <?php include "footer-nav.php"; ?>

    <script>
        const input = document.getElementById('displayNameInput');
        const btn = document.getElementById('saveDisplayNameBtn');
        const originalName = input.value;

        function isValidDisplayName(name) {
            return /^[a-zA-Z0-9_\s]{1,24}$/.test(name);
        }

        btn.addEventListener('click', () => {
            const newName = input.value.trim();
            if (newName === originalName) return;
            if (!isValidDisplayName(newName)) return;

            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            const hiddenInput = document.createElement('input');
            hiddenInput.name = 'displayname';
            hiddenInput.value = newName;
            form.appendChild(hiddenInput);
            document.body.appendChild(form);
            form.submit();
        });

        const ageCategories = <?= json_encode($ageCategories) ?>;
        let currentIndex = <?= $currentIndex ?>;

        const ageLabel = document.getElementById('ageLabel');
        const ageRange = document.getElementById('ageRange');
        const ageInput = document.getElementById('ageCategoryInput');

        const prevBtn = document.getElementById('prevAgeBtn');
        const nextBtn = document.getElementById('nextAgeBtn');

        function updateAgeDisplay() {
            ageLabel.textContent = ageCategories[currentIndex].label;
            ageRange.textContent = ageCategories[currentIndex].range;
            ageInput.value = ageCategories[currentIndex].label;
        }

        prevBtn.addEventListener('click', () => {
            currentIndex = (currentIndex - 1 + ageCategories.length) % ageCategories.length;
            updateAgeDisplay();
        });

        nextBtn.addEventListener('click', () => {
            currentIndex = (currentIndex + 1) % ageCategories.length;
            updateAgeDisplay();
        });

        lucide.createIcons();

        function setupPasswordToggle(inputId, btnId, duration = 6000) {
            const input = document.getElementById(inputId);
            const btn = document.getElementById(btnId);

            let timeout;

            btn.addEventListener('click', () => {
                input.type = 'text';
                clearTimeout(timeout);
                timeout = setTimeout(() => {
                    input.type = 'password';
                }, duration);
            });

            btn.addEventListener('mouseleave', () => input.type = 'password');
            btn.addEventListener('touchend', () => input.type = 'password');
        }

        setupPasswordToggle('new_password', 'showNewBtn', 6000);
        setupPasswordToggle('repeat_password', 'showRepeatBtn', 6000);

        const form = document.getElementById('passwordForm');
        form.addEventListener('submit', (e) => {
            let valid = true;
            const curr = form.querySelector('input[name="current_password"]');
            const newPass = document.getElementById('new_password');
            const repeatPass = document.getElementById('repeat_password');

            if (newPass.value !== repeatPass.value || newPass.value.length < 8) {
                valid = false;
                newPass.classList.add('border', 'border-red-500');
                repeatPass.classList.add('border', 'border-red-500');
                setTimeout(() => {
                    newPass.classList.remove('border', 'border-red-500');
                    repeatPass.classList.remove('border', 'border-red-500');
                }, 6000);
            }

            if (curr.value === '') {
                valid = false;
                curr.classList.add('border', 'border-red-500');
                setTimeout(() => curr.classList.remove('border', 'border-red-500'), 6000);
            }

            if (!valid) e.preventDefault();
        });
    </script>
</body>

<script>
    lucide.createIcons();
</script>