<?php
session_start();

ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1);
ini_set('session.use_strict_mode', 1);

if (!isset($_SESSION['csrf_tokens'])) {
    $_SESSION['csrf_tokens'] = [];
}

function generate_csrf_token()
{
    $token = bin2hex(random_bytes(32));
    $_SESSION['csrf_tokens'][$token] = [
        'created' => time(),
        'used' => false
    ];

    foreach ($_SESSION['csrf_tokens'] as $existing_token => $data) {
        if (time() - $data['created'] > 3600) {
            unset($_SESSION['csrf_tokens'][$existing_token]);
        }
    }

    return $token;
}

function validate_csrf($token)
{
    if (!isset($_SESSION['csrf_tokens'][$token])) {
        return false;
    }

    $csrf_data = $_SESSION['csrf_tokens'][$token];

    if (time() - $csrf_data['created'] > 900) {
        unset($_SESSION['csrf_tokens'][$token]);
        return false;
    }

    if ($csrf_data['used']) {
        unset($_SESSION['csrf_tokens'][$token]);
        return false;
    }

    $_SESSION['csrf_tokens'][$token]['used'] = true;
    return true;
}

function get_user_ip()
{
    static $cached_ip = null;

    if ($cached_ip !== null) {
        return $cached_ip;
    }

    $ip_headers = [
        'HTTP_CF_CONNECTING_IP',
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
        'REMOTE_ADDR'
    ];

    foreach ($ip_headers as $header) {
        if (empty($_SERVER[$header])) {
            continue;
        }

        $ips = explode(',', $_SERVER[$header]);

        foreach ($ips as $ip) {
            $ip = trim($ip);

            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                $cached_ip = $ip;
                return $ip;
            }
        }
    }

    $fallback = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $cached_ip = filter_var($fallback, FILTER_VALIDATE_IP) ? $fallback : '0.0.0.0';

    return $cached_ip;
}

function get_client_fingerprint()
{
    $ip  = get_user_ip();
    $ua  = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    $lang = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
    return hash("sha256", $ip . '|' . $ua . '|' . $lang);
}

function generate_secure_captcha($length = 4)
{
    $characters = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghjkmnpqrstuvwxyz23456789';
    $captcha = '';
    $max = strlen($characters) - 1;

    for ($i = 0; $i < $length; $i++) {
        $captcha .= $characters[random_int(0, $max)];
    }

    return $captcha;
}

function validate_captcha($input)
{
    if (!isset($_SESSION['captcha_code']) || !isset($_SESSION['captcha_time'])) {
        return false;
    }

    if (time() - $_SESSION['captcha_time'] > 300) {
        unset($_SESSION['captcha_code']);
        unset($_SESSION['captcha_time']);
        return false;
    }

    return hash_equals($_SESSION['captcha_code'], $input);
}

$_SESSION['captcha_code'] = generate_secure_captcha();
$_SESSION['captcha_time'] = time();

$dir_users = __DIR__ . "/users/";
if (!is_dir($dir_users)) mkdir($dir_users, 0700, true);

$forbidden_usernames = ['rankly', 'admin', 'administrator', 'root'];

function safe_post($key)
{
    $value = $_POST[$key] ?? '';
    return trim(htmlspecialchars($value, ENT_QUOTES, 'UTF-8'));
}

function sanitize_username($username)
{
    return preg_replace('/[^a-zA-Z0-9_\p{L}]/u', '', $username);
}

function json_response($status, $message = null)
{
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => $status, 'message' => $message], JSON_UNESCAPED_UNICODE);
    exit;
}

function generate_secure_user_id($length = 16)
{
    return bin2hex(random_bytes($length));
}

function get_secret_key()
{
    $path = __DIR__ . "/users/secret_key.php";
    if (!file_exists($path)) {
        die("Secret key file missing!");
    }
    $data = include $path;
    return $data['key'] ?? null;
}

function encrypt_data($plaintext)
{
    $key = get_secret_key();
    if (strlen($key) !== 32) {
        throw new Exception("Invalid key length");
    }

    $iv = random_bytes(16);
    $tag = '';

    $ciphertext = openssl_encrypt(
        $plaintext,
        "aes-256-gcm",
        $key,
        OPENSSL_RAW_DATA,
        $iv,
        $tag,
        "",
        16
    );

    if ($ciphertext === false) {
        throw new Exception("Encryption failed");
    }

    return base64_encode($iv . $tag . $ciphertext);
}

function decrypt_data($encoded)
{
    $key = get_secret_key();
    if (strlen($key) !== 32) {
        return false;
    }

    $data = base64_decode($encoded);
    if ($data === false || strlen($data) < 33) {
        return false;
    }

    $iv = substr($data, 0, 16);
    $tag = substr($data, 16, 16);
    $ciphertext = substr($data, 32);

    $plaintext = openssl_decrypt(
        $ciphertext,
        "aes-256-gcm",
        $key,
        OPENSSL_RAW_DATA,
        $iv,
        $tag
    );

    return $plaintext !== false ? $plaintext : false;
}

function find_user_by_username($username)
{
    global $dir_users;

    // اول از ایندکس جستجو کن
    $index_file = $dir_users . 'username_index.json';
    if (file_exists($index_file)) {
        $index = json_decode(file_get_contents($index_file), true) ?? [];
        if (isset($index[$username])) {
            $user_id = $index[$username];
            $auth_file = $dir_users . $user_id . "/auth.enc";

            if (file_exists($auth_file)) {
                $dec = decrypt_data(file_get_contents($auth_file));
                if ($dec) {
                    list($u, $h) = explode("|", $dec, 2);
                    if ($u === $username) {
                        return ['user_id' => $user_id, 'hash' => $h];
                    }
                }
            }
        }
    }

    // اگر از ایندکس پیدا نشد، جستجوی خطی
    foreach (glob($dir_users . "*/auth.enc") as $file) {
        $dec = decrypt_data(file_get_contents($file));
        if (!$dec) continue;

        list($u, $h) = explode("|", $dec, 2);
        if ($u === $username) {
            $user_id = basename(dirname($file));
            add_user_to_index($username, $user_id);
            return ['user_id' => $user_id, 'hash' => $h];
        }
    }

    return null;
}

function username_exists($username)
{
    return find_user_by_username($username) !== null;
}

function email_exists($email)
{
    global $dir_users;

    $email = strtolower($email);
    foreach (glob($dir_users . "*/data.json") as $file) {
        $info = json_decode(file_get_contents($file), true);
        if (isset($info['email']) && strtolower($info['email']) === $email) {
            return true;
        }
    }
    return false;
}

function ip_or_fingerprint_exists($ip, $fingerprint)
{
    global $dir_users;

    foreach (glob($dir_users . "*/data.json") as $file) {
        $info = json_decode(file_get_contents($file), true);

        if (isset($info['last_ip']) && $info['last_ip'] === $ip) return true;
        if (isset($info['now_ip']) && $info['now_ip'] === $ip) return true;
        if (!empty($info['fingerprint']) && $info['fingerprint'] === $fingerprint) return true;
    }

    return false;
}

function add_user_to_index($username, $user_id)
{
    global $dir_users;

    $index_file = $dir_users . 'username_index.json';
    $index = [];

    if (file_exists($index_file)) {
        $index = json_decode(file_get_contents($index_file), true) ?? [];
    }

    $index[$username] = $user_id;
    file_put_contents($index_file, json_encode($index, JSON_PRETTY_PRINT | LOCK_EX));
}

function rrmdir($dir)
{
    if (!is_dir($dir)) return;

    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        is_dir($path) ? rrmdir($path) : unlink($path);
    }
    rmdir($dir);
}

$max_attempts = 5;
$lock_time = 300;

if (!isset($_SESSION['login_attempts'])) {
    $_SESSION['login_attempts'] = 0;
    $_SESSION['lock_until'] = 0;
}

if (isset($_GET['login'])) {

    if (time() < $_SESSION['lock_until']) {
        json_response('error', 'Access is restricted for a while!');
    }

    $csrf = safe_post('csrf_token');
    if (!validate_csrf($csrf)) {
        json_response('error', 'Invalid security token');
    }

    $username = sanitize_username(safe_post('username'));
    $password = safe_post('password');

    if ($username === "" || $password === "") {
        json_response('error', 'Please fill in all fields');
    }

    if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        json_response('error', 'The username or password is incorrect');
    }

    $user_data = find_user_by_username($username);
    $login_failed = true;

    if ($user_data && password_verify($password, $user_data['hash'])) {
        $login_failed = false;
        $user_id = $user_data['user_id'];
    }

    if ($login_failed) {
        $_SESSION['login_attempts']++;
        if ($_SESSION['login_attempts'] >= $max_attempts) {
            $_SESSION['lock_until'] = time() + $lock_time;
            $_SESSION['login_attempts'] = 0;
            json_response('error', 'Please try again in 5 minutes!');
        }

        sleep(2);
        json_response('error', 'The username or password is incorrect');
    }

    $_SESSION['login_attempts'] = 0;
    $_SESSION['lock_until'] = 0;
    $_SESSION['user_id'] = $user_id;

    session_regenerate_id(true);

    $dataFile = $dir_users . $user_id . "/data.json";

    if (!file_exists($dataFile)) {
        session_destroy();
        json_response('error', 'User data not found');
    }

    $data = json_decode(file_get_contents($dataFile), true);
    if ($data === null) {
        session_destroy();
        json_response('error', 'Invalid user data');
    }

    $data['last_login'] = date("Y-m-d H:i:s");
    $current_ip = get_user_ip();
    $data['last_ip'] = $data['now_ip'];
    $data['now_ip'] = $current_ip;
    $data['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? $data['user_agent'];

    if (!file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX)) {
        json_response('error', 'Failed to update user data');
    }

    json_response('success', 'Login was successful!');
}

if (isset($_GET['register'])) {

    $username = sanitize_username(safe_post('username'));
    $password = safe_post('password');
    $email    = safe_post('email');
    $ip       = get_user_ip();
    $fingerprint = get_client_fingerprint();

    if ($username === "" || $password === "" || $email === "") {
        json_response('error', 'Please complete all fields.');
    }

    $csrf = safe_post('csrf_token');
    if (!validate_csrf($csrf)) {
        json_response('error', 'Invalid security token');
    }

    sleep(1);

    if (in_array(mb_strtolower($username), $forbidden_usernames)) {
        json_response('error', 'This username is not allowed');
    }

    if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        json_response('error', 'Username must be 3-20 characters, contain only letters, numbers, or _.');
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        json_response('error', 'The email is not valid');
    }

    $email_domain = explode('@', $email)[1];
    if (!checkdnsrr($email_domain, 'MX') && !checkdnsrr($email_domain, 'A')) {
        json_response('error', 'The email domain is not valid');
    }

    if (strlen($password) < 8) {
        json_response('error', 'Password must be at least 8 characters long');
    }

    if (username_exists($username)) {
        json_response('error', 'This username is already registered');
    }

    if (email_exists($email)) {
        json_response('error', 'This email has already been used!');
    }

    if (ip_or_fingerprint_exists($ip, $fingerprint)) {
        json_response('error', 'There was an error registering. Please try again later!');
    }

    $id = generate_secure_user_id();
    $userDir = $dir_users . $id;

    if (!mkdir($userDir, 0700, true)) {
        json_response('error', 'There was an error registering. Please try again later!');
    }

    $data = [
        "id"          => $id,
        "username"    => $username,
        "displayname" => $username,
        "email"       => strtolower($email),
        "age"         => "",
        "registered"  => date("Y-m-d H:i:s"),
        "last_login"  => "",
        "language"    => "en",
        "last_ip"     => $ip,
        "now_ip"      => $ip,
        "user_agent"  => $_SERVER['HTTP_USER_AGENT'] ?? "",
        "fingerprint" => $fingerprint
    ];

    $dataFile = $userDir . "/data.json";

    if (!file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX)) {
        rrmdir($userDir);
        json_response('error', 'There was an error registering. Please try again later!');
    }

    $hash = password_hash($password, PASSWORD_DEFAULT);
    $encrypted = encrypt_data($username . "|" . $hash);

    if (!file_put_contents($userDir . "/auth.enc", $encrypted, LOCK_EX)) {
        rrmdir($userDir);
        json_response('error', 'There was an error registering. Please try again later!');
    }

    add_user_to_index($username, $id);

    json_response('success', 'Registration was successful!');
}
?>

<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="/images/favicon.png">
    <title>Log in to Rankly</title>
</head>

<body class="h-screen flex items-center justify-center">
    <?php include "header.php"; ?>

    <?php if (!isset($_SESSION['user_id'])): ?>

        <div class="flex flex-col items-center justify-start w-full">

            <div id="welcome-screen" class="flex flex-col items-center text-center gap-4 w-full px-1">

                <div class="max-w-3xl w-full mt-5 p-6 relative">
                    <div id="introText" class="text-white text-lg transition-opacity duration-500"></div>

                    <button id="langSwitch" class="flex text-[#ffffff85] mt-4 text-[14px] items-center">
                        <i data-lucide="languages" class="w-6 h-6 mr-1"></i>
                        <span id="langLabel">Persian فارسی</span>
                    </button>
                </div>

                <button id="btn-show-login"
                    class="w-[90%] mt-5 bg-white text-black py-2 rounded-lg text-[18px]">
                    Sign In
                </button>

                <div class="flex items-center w-[90%]">
                    <div class="flex-grow border-t border-[#555]"></div>
                    <span class="px-3 text-[#999]">or</span>
                    <div class="flex-grow border-t border-[#555]"></div>
                </div>

                <button id="btn-show-register"
                    class="w-[90%] bg-white text-black py-2 rounded-lg text-[18px]">
                    Sign Up
                </button>

                <p class="text-[#555] mt-10 text-sm">© 2025 Rankly • Developer: Kian Rastegar</p>
            </div>

            <div id="login-form-wrapper" class="hidden w-full px-5">

                <div class="w-full flex items-center justify-between border-y border-[#444] py-3 mb-5">
                    <span class="text-white text-[20px] font-medium">Login To Account</span>
                    <button id="back-from-login" class="text-white">
                        <i data-lucide="arrow-right" class="w-6 h-6"></i>
                    </button>
                </div>

                <form id="login-form" class="flex flex-col gap-3">
                    <input type="text" name="username" placeholder="Username"
                        class="p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] rtl" />

                    <input type="password" name="password" placeholder="Password"
                        class="p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] rtl" />

                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">

                    <button type="submit" name="login"
                        class="bg-white text-black rounded p-2 font-semibold">
                        Login
                    </button>
                </form>
            </div>

            <div id="register-form-wrapper" class="hidden w-full px-5">

                <div class="w-full flex items-center justify-between border-y border-[#444] py-3 mb-5">
                    <span class="text-white text-[20px] font-medium">Create Account</span>
                    <button id="back-from-register" class="text-white">
                        <i data-lucide="arrow-right" class="w-6 h-6"></i>
                    </button>
                </div>

                <form id="register-form" class="flex flex-col gap-3">

                    <input type="text" name="username" placeholder="Username"
                        class="p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] rtl" />

                    <input type="text" name="email" placeholder="Email"
                        class="p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] rtl" />

                    <input type="password" name="password" placeholder="Password"
                        class="p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] rtl" />

                    <div id="captcha-section" class="flex flex-col items-center gap-1">
                        <div class="flex items-center gap-2 w-full flex-nowrap">
                            <div id="captcha-display"
                                class="flex-1 max-w-[85px] pl-1 h-[49px] pt-1 bg-gradient-to-r from-blue-600 to-blue-700 rounded-lg 
            flex items-center justify-center text-white font-bold text-xl tracking-widest select-none cursor-default">
                                <svg id="captcha-svg" width="100" height="30"></svg>
                            </div>
                            <input type="text" id="captcha-input"
                                maxlength="4"
                                placeholder="Enter The Code"
                                class="flex-1 w-[100%] p-3 rounded-md bg-[#181818] text-white text-lg placeholder-[#aaa] focus:outline-none focus:bg-[#1c1c1c] text-center rtl direction-ltr"
                                style="letter-spacing: 2px;" />
                        </div>
                        <span class="mt-2 text-center text-[12px] text-[#ffffff50]">Rankly Captcha | Are You Human?</span>
                    </div>

                    <button type="button" id="captcha-submit-btn"
                        class="bg-gray-500 text-white rounded p-2 font-semibold flex items-center justify-center gap-2 transition-all duration-300 cursor-not-allowed">
                        <span>Check Captcha</span>
                    </button>

                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">

                    <button type="submit" name="register" id="register-submit-btn" class="flex justify-center bg-white text-center text-black rounded p-2 font-semibold hidden">Submit</button>
                </form>
            </div>
            <div id="message-box" class="text-sm mb-2 text-red-500"></div>
        <?php
    else:
        header("Location: profile.php");
        exit;
    endif;
        ?>

        <?php include "footer-nav.php"; ?>

        <script>
            document.addEventListener("DOMContentLoaded", () => {
                const welcome = document.getElementById("welcome-screen");
                const loginWrap = document.getElementById("login-form-wrapper");
                const registerWrap = document.getElementById("register-form-wrapper");

                const btnShowLogin = document.getElementById("btn-show-login");
                const btnShowRegister = document.getElementById("btn-show-register");
                const backLogin = document.getElementById("back-from-login");
                const backRegister = document.getElementById("back-from-register");

                const loginForm = document.getElementById("login-form");
                const registerForm = document.getElementById("register-form");

                const messageBox = document.getElementById("message-box");

                function showMessage(text, type = "error") {
                    messageBox.textContent = text;
                    messageBox.classList.remove("text-red-500", "text-green-500");
                    messageBox.classList.add(type === "error" ? "text-red-500" : "text-green-500");
                }

                function clearMessage() {
                    messageBox.textContent = "";
                    messageBox.classList.remove("text-red-500", "text-green-500");
                }

                btnShowLogin.addEventListener("click", () => {
                    clearMessage();
                    welcome.classList.add("hidden");
                    loginWrap.classList.remove("hidden");
                });
                btnShowRegister.addEventListener("click", () => {
                    clearMessage();
                    welcome.classList.add("hidden");
                    registerWrap.classList.remove("hidden");
                });
                backLogin.addEventListener("click", () => {
                    clearMessage();
                    loginWrap.classList.add("hidden");
                    welcome.classList.remove("hidden");
                });
                backRegister.addEventListener("click", () => {
                    clearMessage();
                    registerWrap.classList.add("hidden");
                    welcome.classList.remove("hidden");
                });

                function handleFormSubmit(form, endpoint) {
                    form.addEventListener("submit", e => {
                        e.preventDefault();
                        const formData = new FormData(form);

                        fetch(endpoint, {
                                method: "POST",
                                body: formData
                            })
                            .then(res => res.json())
                            .then(data => {
                                clearMessage();
                                if (data.status === "success") {
                                    showMessage(data.message || "successful", "info");
                                    if (endpoint.includes("login")) setTimeout(() => {
                                        window.location.href = "profile.php";
                                    }, 500);
                                    if (endpoint.includes("register")) setTimeout(() => {
                                        registerWrap.classList.add("hidden");
                                        loginWrap.classList.remove("hidden");
                                    }, 500);
                                } else {
                                    showMessage(data.message || "error", "error");
                                }
                            })
                            .catch(() => {
                                showMessage("error", "error");
                            });
                    });
                }

                handleFormSubmit(loginForm, "login.php?login=1");
                handleFormSubmit(registerForm, "login.php?register=1");
            });

            document.addEventListener("DOMContentLoaded", () => {

                const captchaInput = document.getElementById('captcha-input');
                const captchaSubmitBtn = document.getElementById('captcha-submit-btn');
                const registerSubmitBtn = document.getElementById('register-submit-btn');
                const captchaDisplay = document.getElementById('captcha-display');
                const refreshCaptchaBtn = document.getElementById('refresh-captcha');
                const captchaSection = document.getElementById('captcha-section');

                captchaInput.addEventListener('input', function(e) {
                    this.value = this.value.replace(/[^a-zA-Z0-9]/g, '');
                    if (this.value.length === 4) {
                        validateCaptchaInput();
                    }
                    updateCaptchaButtonState();
                });

                function updateCaptchaButtonState() {
                    if (captchaInput.value.length === 4) {
                        captchaSubmitBtn.classList.remove('bg-gray-500', 'cursor-not-allowed');
                        captchaSubmitBtn.classList.add('bg-black', 'hover:bg-black', 'cursor-pointer');
                    } else {
                        captchaSubmitBtn.classList.remove('bg-black', 'hover:bg-black', 'cursor-pointer');
                        captchaSubmitBtn.classList.add('bg-gray-500', 'cursor-not-allowed');
                    }
                }

                function validateCaptchaInput() {
                    const enteredCode = captchaInput.value;
                    const actualCode = captchaDisplay.textContent.trim();

                    if (enteredCode === actualCode) {
                        captchaSubmitBtn.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-check"><path d="M20 6 9 17l-5-5"/></svg>';
                        captchaSubmitBtn.classList.remove('bg-black');
                        captchaSubmitBtn.classList.add('bg-black');

                        setTimeout(() => {
                            captchaSection.classList.add('hidden');
                            registerSubmitBtn.classList.remove('hidden');
                            registerSubmitBtn.classList.add('flex');
                        }, 1000);
                    } else {
                        captchaSubmitBtn.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-x"><path d="M18 6 6 18"/><path d="m6 6 12 12"/></svg>';
                        captchaSubmitBtn.classList.remove('bg-black');
                        captchaSubmitBtn.classList.add('bg-black');

                        setTimeout(() => {
                            captchaSubmitBtn.innerHTML = '<span>Check Captcha</span>';
                            captchaSubmitBtn.classList.remove('bg-black');
                            captchaSubmitBtn.classList.add('bg-gray-500');
                            captchaInput.value = '';
                            updateCaptchaButtonState();
                        }, 1500);
                    }
                }

                captchaSubmitBtn.addEventListener('click', function() {
                    if (captchaInput.value.length === 4) {
                        validateCaptchaInput();
                    }
                });

                captchaDisplay.addEventListener('copy', (e) => {
                    e.preventDefault();
                });

                captchaDisplay.addEventListener('contextmenu', (e) => {
                    e.preventDefault();
                });
            });

            (function() {
                const code = "<?php echo $_SESSION['captcha_code']; ?>";

                const svg = document.getElementById("captcha-svg");
                const ns = "http://www.w3.org/2000/svg";

                const svgWidth = 80;
                const svgHeight = 40;
                const charWidth = 20;
                const totalWidth = code.length * charWidth;

                const startX = (svgWidth - totalWidth) / 4;

                for (let i = 0; i < 25; i++) {
                    const circle = document.createElementNS(ns, "circle");
                    circle.setAttribute("cx", Math.random() * svgWidth);
                    circle.setAttribute("cy", Math.random() * svgHeight);
                    circle.setAttribute("r", Math.random() * 1.5);
                    circle.setAttribute("fill", "rgba(255,255,255,0.2)");
                    svg.appendChild(circle);
                }

                const placeholder = document.createElementNS(ns, "text");
                placeholder.setAttribute("x", "50%");
                placeholder.setAttribute("y", "50%");
                placeholder.setAttribute("dominant-baseline", "middle");
                placeholder.setAttribute("text-anchor", "middle");
                placeholder.setAttribute("font-family", "'Courier New', monospace");
                placeholder.setAttribute("font-size", "20");
                placeholder.setAttribute("fill", "white");

                const encoded = btoa(code);
                placeholder.textContent = encoded;
                svg.appendChild(placeholder);

                setTimeout(() => {
                    svg.removeChild(placeholder);

                    let x = startX;
                    [...code].forEach(char => {
                        const t = document.createElementNS(ns, "text");
                        t.setAttribute("x", x + (Math.random() * 2 - 1));
                        t.setAttribute("y", 22 + (Math.random() * 2 - 1));
                        t.setAttribute("font-family", "'Courier New', monospace");
                        t.setAttribute("font-size", "22");
                        t.setAttribute("fill", "white");
                        t.textContent = char;

                        svg.appendChild(t);
                        x += charWidth;
                    });

                }, 10);

            })();

            let currentLang = 'en';
            let translations = {};

            async function loadLang(lang) {
                const response = await fetch(`texts/${lang}.lang`);
                const text = await response.text();
                const dict = {};
                text.split('\n').forEach(line => {
                    if (line.trim() && line.includes('=')) {
                        const [key, value] = line.split('=');
                        dict[key.trim()] = value.trim();
                    }
                });
                return dict;
            }

            function displayText() {
                const textContainer = document.getElementById('introText');
                textContainer.classList.add('opacity-0');

                setTimeout(() => {
                    textContainer.innerText = translations['intro'];
                    textContainer.classList.remove('opacity-0');

                    if (currentLang === 'fa') {
                        textContainer.setAttribute('dir', 'rtl');
                        textContainer.classList.add('text-right', 'text-[15px]', 'vazir_font');
                        textContainer.classList.remove('text-left', 'text-[13.9px]', 'inter_font');
                    } else {
                        textContainer.setAttribute('dir', 'ltr');
                        textContainer.classList.add('text-left', 'text-[13.9px]', 'inter_font');
                        textContainer.classList.remove('text-right', 'text-[15px]', 'vazir_font');
                    }
                }, 200);
            }

            async function switchLang() {
                currentLang = currentLang === 'en' ? 'fa' : 'en';
                translations = await loadLang(currentLang);
                displayText();
                const label = document.getElementById('langLabel');
                label.innerText = currentLang === 'en' ? 'Persian فارسی' : 'English انگلیسی';
            }

            window.addEventListener('DOMContentLoaded', async () => {
                translations = await loadLang(currentLang);
                displayText();

                document.getElementById('langSwitch').addEventListener('click', switchLang);
            });
        </script>
</body>

<script>
    lucide.createIcons();
</script>